/* Motors.nc -- Interface for control of Robot move and turn motors.
 *
 *  Commands: forward(), backward(), rotateCW(), rotateCCW().
 *  Command: stopMotors() turns off motors.
 *  Speed is an 8-bit number from 1-255 which should be cm/sec.
 *  Distance is an 8-bit number from 0-255 which should be cm.
 *  Angle is an 8-bit number from 0-180 which should be 2deg incr.
 *  All input parameters are advisory, actual results are available
 *	 from the accessors, or may be posted to a global status area.
 *
 *  This interface will probably be used in conjunction with the
 *	 SensorStatus interface to report motion completion events, and
 *	 SensorControl interface to initialize, setup, and monitor.
 *
 *	Nominally for differential two drive motor setup,
 *   but could be used for Ackerman steering/motor too...
 *  Probably using PWMcontrol of modified "hobby-servo" motors
 *	 using Timer1A,B outputs defined in HPLT1pwmM.nc:
 * 		PWM1Aout   PB5		MotorA (left) PWM
 *		PWM1Bout   PB6		MotorB (right) PWM
 *
 * Authors:		M.Schippling
 */

//  setDiff() sets a speed differential between the motors.

interface Motors
{
  /** stopMotors()
   *    Stop all the motors and update status values.
   *  @param sig -- if true send a statusReady() signal.
   *  @return nada.
   */
  command void stopMotors( bool sig );
  
  /** forward()
   *    Start robot moving forward for the given distance and speed.
   *	Both parameters are advisory, actuals are reported after stop.
   *  @param d -- distance in cm (255 max).
   *  @param s -- speed in cm/s (255 max).
   *  @return nada.
   */
  command void forward( int16_t d, uint8_t s );

  /** backward()
   *    Start robot moving backward for the given distance and speed.
   *	 ...d value can be positive or negative...
   *	Both parameters are advisory, actuals are reported after stop.
   *  @param d -- distance in cm (255 max).
   *  @param s -- speed in cm/s (255 max).
   *  @return nada.
   */
  command void backward( int16_t d, uint8_t s );

  /** rotateCW()
   *    Start robot rotating/turning clockwise
   *	 for the given angle and speed.
   *	 ... nominal CW angle is negative but the 'a' arg
   *         can be positive or negative...
   *	Both parameters are advisory, actuals are reported after stop.
   *  @param a -- angle in 2deg increments (180 -- 360deg -- max).
   *  @param s -- speed in cm/s (255 max).
   *  @return nada.
   */
  command void rotateCW( int16_t a, uint8_t s );

  /** rotateCCW()
   *    Start robot rotating/turning counter-clockwise
   *	 for the given angle and speed.
   *	Both parameters are advisory, actuals are reported after stop.
   *  @param a -- angle in 2deg increments (180 -- 360deg -- max).
   *  @param s -- speed in cm/s (255 max).
   *  @return nada.
   */
  command void rotateCCW( int16_t a, uint8_t s );

  /** jogLeft()
   *	Jog the robot to the left by amount given.
   *    nominally: Change MotorA speed +/- .
   *  @param s -- +/- speed increment
   *  @return nada.
   */
  command void jogLeft( int8_t s );

  /** jogRight()
   *	Jog the robot to the right by amount given.
   *    nominally: Change MotorB speed +/- .
   *  @param s -- +/- speed increment
   *  @return nada.
   */
  command void jogRight( int8_t d );

  /** enableSpdCntl()
   *    En/Dis-able encoder driven speed control in check*Spd() methods.
   *	TRUE enables, FALSE disables.
   *	May be used to take motor control during Target Tracking.
   *  @return nada.
   */
  command void enableSpdCntl( bool en );

  /** getDistance()
   *  @return the distance in CM traveled during the last motion.
   *   Will be negative if travel was backward.
   *   Also returns the residual difference when doing a turn.
   */
  command int16_t getDistance();
  
  /** getAngle()
   *  @return the angle in degrees traveled during the last motion.
   *   Will be negative if travel was CW.
   *   Also returns the residual difference when doing a fwd/bkd move
   */
  command int16_t getAngle();
  
  /** getSpeed()
   *  @return the speed of the last motion.
   */
  command uint8_t getSpeed();
  
}
