/* HSMotorsM.nc -- Code to control Robot move and turn motors.
 *
 *   For modified Hobby Servo motors including added encoders.
 *   MotorA is the Left and MotorB is the Right 
 *    in a differential two drive motor setup.
 *   Assumes that they are mounted mirror image so have to turn
 *    in opposite directions to go fwd/bkd.
 *  Uses Frequency Generator outputs on PWM1a,b (ports PB5,6),
 *   to drive PWMcontrol of modified "hobby-servo" motors
 *	 using Timer1A,B outputs defined in HPLT1pwmM.nc:
 * 		PWM1Aout   PB5		MotorA (left) PWM
 *		PWM1Bout   PB6		MotorB (right) PWM
 *
 *  Move Commands: forward(), backward(), rotateCW(), rotateCCW().
 *  Stop Command: stopMotors() turns off motors.
 *	Status signal MotorStatus.statusReady() -- motion complete notify
 *  Speed is an 8-bit number from 1-255 which should be cm/sec.
 *  Distance is an 16-bit number from 0-255 which should be cm.
 *  Angle is an 16-bit number from 0-360 which should be 1deg incr.
 *  All input parameters are advisory, actual results are available
 *	 from the accessors, and are posted to the global control struct.
 *	 Returned values will indicate the direction with the sign:
 *		Forward = plus, Backward = minus
 *		Counter-clockwise = plus, Clockwise = minus
 *
 *  The Motor interface is used in conjunction with the
 *	 SensorStatus interface to report motion completion events.
 *	 SensorControl interface provides initialize, setup, and monitor.
 *	 Encoder SensorStatus interface is used to get encoder ticks.
 *	 Timer interface is used as a watchdog in case the encoders fail.
 *
 * Function:
 *	User calls init() to get the ball rolling.
 *	In normal operation:
 *	 User calls one of the Move commands;
 *		Speed, distance, and timeout are calculated,
 *		Timeout timer and encoders are started,
 *		Motors are started at appropriate speed with setPW().
 *	  Encoder statusReady() signals update the motor speeds as needed.
 *	  Either the encoder tick count is complete or timer timeout fires,
 *		stopMotors is called:
 *		  Motors are stopped;
 *		  Timer is stopped just in case...
 *		  Completion distances are updated;
 *		  User is notified via MotorStatus.statusReady()
 *	...or...
 *	  User calls stopMotors() -- because bumpers fired or something...
 *		The same stopMotors() behavior can result in
 *			 MotorStatus.statusReady() user notification callback.
 *
 * Authors:		M.Schippling
**/
includes RoboMsg;  /* the basic request and status messages */

module HSMotorsM
{
  provides interface Motors;
  provides interface SensorControl as MotorControl;
  provides interface SensorStatus as MotorStatus;
  
  uses
  {
  	 interface HPLT1pwm as Pwm;
  	 interface SensorStatus as EncoderStatus;
  	 interface DataStore;
  	 interface Timer;
  	 interface Leds;
  }
}
implementation
{
	//#define CALADJWINDOW	// to enlarge dither reject window
	//#define ERRORWINDOW	// to reject out of band signals
	#define DOSPEEDCONTROL	// to actually try to control motor speeds
	#define SENDENCVAL	// put encoder counts in statp for debug

	//#define LEDRIGHTLOCK // turn on Red LED with right wheel spdlock
	//#define LEDLEFTLOCK // turn on Red LED with left wheel spdlock

	// these get set to globals shared by RobotM
	RoboStatusMsg *statp;	// outgoing status message body
	RoboControl *cntlp;		// local control values
	
	// total number of Encoder clicks to allow during motion
	uint16_t enClicks;
	// number of encoder clicks for each motor
	//  note limited to ~127 cm travel at 2 click/cm
	uint8_t mAclicks = 0;
	uint8_t mBclicks = 0;
	// desired encoder tick/click count for current speed
	uint8_t enTcount;
	// "error bars" for encoder signal tick/click
	uint8_t minEticks, maxEticks;
	// tick/click slop window to prevent dithering, was "incr"
	uint8_t minTWin, maxTWin;
	// last tick count values we got for each encoder
	uint8_t lAcount, lBcount;
	
	// status bits, see MOTORSTATUS enum
	uint8_t motorstat = 0;	
	
	// current pulse-width settings for each motor
	uint8_t mApw = 0;
	uint8_t mBpw = 0;
	
	// Speed is specified as an 8-bit number from 4-255 which sets
	//   the relative On/Off pulse-width ratio for both motors.
	// The initial duty-cycle pw values are calculated from below.
	enum
	{
		MAXCCW		=	150,	// max CCW rotation count
		MINCCW		=	93,		// min CCW rotation count
		S0			=	90,		// ~1.5ms nominal servo off
		MINCW		=	87,		// min CW rotation count
		MAXCW		=	30,		// max CW rotation count
		PULSEROT	=	44,		// encoder pulses per wheel rotation
		WHEELCIRC	=	22,		// circumference of wheel in CM
		ROBOTCIRC	=	27,		// circum of circle robot makes in CM
								// ~6.6 degree/encoder-click
		MANGLEFP16	=	107,	// multiplier for x16 fixp angle calcs
		
		// relative speed to duty cycle range divisor (...val=4...)
		SD	=	(512/(MAXCCW-MAXCW)),
		// encoder pulses per CM travel
		PC	=	(PULSEROT/WHEELCIRC),
		
		// milli-sec in 1 second...for convenience
		MS	= 1000,
		// number of Timer() ticks per second
		BMS = 1024,
		
		// number of encoder clicks to wait for acceleration
		ACCELCLICK = 4,
		// increment value for speed control jogging 
		JOGINCR = 1
	};
	
	// map of relative speed to actual pulse-width settings
	//  for both motors going both directions
	//  nominal forward is CCW motion for motorA (on left side)
	//                and   CW motion for motorB (on right side),
	//    because one is mirror mounted and needs to run "backward"...
	typedef struct
	{
		uint8_t spdset;		// nominal speed setting cm/s, table index
		uint8_t cwAcnt;		// CW MotorA pulse-width count setting
		uint8_t ccwAcnt;	// CCW MotorA pulse-width count setting
		uint8_t cwBcnt;		// CW MotorB pulse-width count setting
		uint8_t ccwBcnt;	// CCW MotorB pulse-width count setting
	} speed_table;
	
	// calculate based on:  encoder pulses per wheel rotation
	//   count -- number of timer ticks between pulses @ each speed
	//		1024 / (pulses/cm * cm/sec)
	//   Xcnts  -- PW settings for each motor@direction
	//		seat'o'pants est based on SD range divisor and spdsetting
	//		nominal 90==0-mid-point >90==CCW, <90==CW rotation
	//    Note that RoboII mtrs/wheels/bats poop at about 25cm/sec
	//	   At 2 clicks per cm, binary-ticks/click -- err range, incr
	//		25cm/sec == 50 click/sec ==  20 bticks/click --  5,  2
	//		10cm/sec == 20 click/sec ==  51 bticks/click -- 12,	 6
	//		 5cm/sec == 10 click/sec == 102 bticks/click -- 25,	12
	speed_table spdtbl[] =
	{
		// NOTE: for 0 speed should shut off encoders and use timeout....
		{   0, S0, S0, S0, S0 },		//  0cm/s -- for calibration
//		{   5, 							//  5cm/s
//			 S0-(10/SD), S0+(10/SD), S0-(10/SD), S0+(10/SD) },
		{   6, 							//  6cm/s
			 S0-(12/SD), S0+(12/SD), S0-(12/SD), S0+(12/SD) },
//		{   7, 							//  7cm/s
//		 	S0-(16/SD), S0+(16/SD), S0-(16/SD), S0+(16/SD) },
//		{  10, 							// 10cm/s
//			 S0-(20/SD), S0+(20/SD), S0-(20/SD), S0+(20/SD) },
		{  12, 							// 12cm/s
			 S0-(25/SD), S0+(25/SD), S0-(25/SD), S0+(25/SD) },
		{  18, 							// 18cm/s
			 S0-(45/SD), S0+(45/SD), S0-(45/SD), S0+(45/SD) },
//		{  20, 							// 20cm/s
//		 	S0-(50/SD), S0+(50/SD), S0-(50/SD), S0+(50/SD) },
		{  24, 							// 24cm/s
			 S0-(60/SD), S0+(60/SD), S0-(60/SD), S0+(60/SD) }
//		{  25, 							// 25cm/s
//			 S0-(60/SD), S0+(60/SD), S0-(60/SD), S0+(60/SD) }

/** original values from above table
	 spd   Acw Acc Bcw Bcc	==
	   FWD:___ Acc Bcw ___
	   BKD:Acw ___ ___ Bcc 
		 0 90  90  90  90 
		 6 87  93  87  93 
		12 84  96  84  96 
		18 79 101  79 101 
		24 75 105  75 105
**/
/** after running with accel control on Juan:
		0 90 90 90 90 
		6 87 93 87 93 
		12 84 96 84 95 
		18 80 98 78 96 
		24 76 98 72 100
**/
/** nominal forward() settings, A-CCW, B-CW:
			0 = PWMsetABpw8( 90, 90 )
			20 = PWMsetABpw8( 102, 78 )
			25 = PWMsetABpw8( 115, 65 )
			
		{0,90,90,90,90 },
		{6,87,93,87,93 },		// reasonable values for Juan
		{12,84,94,83,95},		// FWD:AsBf BKD:AfBs
		{18,80,98,77,98 },		// FWD:AfBs BKD:AfBs
//		{24,75,105,75,105 },	// funky original values...
		{24,67,100,65,105 },	// heuristic values for Juan...
**/	
	};
	
	speed_table *curtbl;		// current speed_table entry
	
	  
	/** init()
	 *  Setup the Timer1A,B for Phase and Frequency correct PWM
	 *   with default duration of 20ms and outputs OFF.
	**/
	command result_t MotorControl.init()
	{
		call Pwm.init();
/**/
		call DataStore.getData(
				MOTORPARAMS, (uint8_t*)spdtbl, sizeof(spdtbl) );
/**/
		
		//call Pwm.setABpw8( 102, 78 );		// force motor test
		return SUCCESS;
	}
	
	command result_t MotorControl.start()
	{
		//Pwm.start(); // aren't none...
		return SUCCESS;
	}
	
	/** Now just HOLD EVERYTHING...
	**/
	command result_t MotorControl.stop()
	{
		call EncoderStatus.doStatus( 0 );
		call Pwm.stopAll();
		call Timer.stop();
		return SUCCESS;
	}
	
	/** set 'global' variable pointers as needed
	 */
	command result_t MotorControl.setGlobals(
  					 TOS_Msg *statmsg,		 // status return
  					 RoboCmdMsg *cmdbuf,	 // current command
  					 RoboControl *cntlbuf )  // internal data
  	{
	  	statp = (RoboStatusMsg*)statmsg->data;
  		cntlp = cntlbuf;
  		return SUCCESS;
  	}
	
/******* NOTE: For the moment the commands specify
		 distance  and angle as milli-seconds of ON time
		 so we'll "calculate backwards". Once things is settled
		 we'll switch to cm and degrees...
********/
	
	/** Figure out encoder ticks,
	 *  Set curtbl pointer to appropriate spdtbl entry based on speed.
	 *  Set global enClicks to final encoder count we want.
	 *  Set min,maxEticks to error range of encoder ticks for speed.
	 *  Start Encoder.
	 *
	 *  @param a -- angle in 2deg increments (180 -- 360deg -- max).
	 *  @param d -- distance in cm (255 max).
	 *  @param s -- speed in cm/s (255 max).
	 */
	void calcEncoder( uint16_t a, uint16_t d, uint8_t s )
	{
		uint8_t i;
		uint32_t ltime;
		
		// set curtbl to point to the speed setting entry in spdtbl
		//  that is just greater than or equal to desired speed.
		//  note: this drops out on last spdtbl entry if not found
		//  		so last entry is the default.
		for( i = 0;
			 i < (sizeof(spdtbl) / sizeof(speed_table)) - 1 ;
			 ++i )
		{
			if( s <= spdtbl[i].spdset )
				break;
		}
		curtbl = &spdtbl[i];
	
		// zero speed is the special calibration setting
		// don't do encoders and default time to a fixed value
		if( s == 0 )
		{
			motorstat &= ~MSspdEn;
			return;
		}
		
#ifdef DOSPEEDCONTROL
		// enable encoder speed control
		motorstat |= MSspdEn;
#endif
		
		//NOTE: need to implement correctly!!!
		// to use real distance rather than time
		
		// calculate total clicks for motorsA+B...
		// time is in ms and total clicks are 4/cm
		//  use 32bits to minimize roundoff error
		// get the total time of move and turn
		ltime = a+d;
		// get the number of clicks we expect for this time
		ltime *= ( curtbl->spdset * (PC * 2) );
		
		// divide by MS -- times are in milli-sec and c/s is seconds
		enClicks = (uint16_t) (ltime / MS);
		
		// calculate desired encoder binary-tick/click count
		//  approx: 6cm/s == 85 ... 24cm/s == 21
		enTcount = BMS/(PC * curtbl->spdset);
		
		// calculate error bars for tick counts
		//  so we can reject garbage signals
#ifdef ERRORWINDOW
		i = (enTcount >> 2);	// 25%
		minEticks = (enTcount - i);
		maxEticks = (enTcount + i);
#else
		minEticks = 1;
		maxEticks = 255;
#endif // not ERRORWINDOW
		
		// calculate adjustment window
#ifdef CALADJWINDOW
		i = (enTcount >> 3);	// 12%
		minTWin = (enTcount - i);
		maxTWin = (enTcount + i);
#else  // just use a fixed window
		minTWin = (enTcount - 1);
		maxTWin = (enTcount + 1);
#endif // not ADJUSTWINDOW
		
		// signup for encoder status signals
		call EncoderStatus.doStatus( 1 );
	}
	
	/** Figure out timeout, clear ticks counter, start Timer
	 *  @param a -- angle in 1deg increments (180 -- 360deg -- max).
	 *  @param d -- distance in cm (255 max).
	 *  @param s -- speed in cm/s (255 max).
	 */
	void calcTimer( uint16_t a, uint16_t d, uint8_t s )
	{
		uint16_t runtime;
		
		// zero speed is the special calibration setting
		// don't do encoders and default time to a fixed value
		if( s == 0 )
		{
			runtime = 10000;
		}
		else
		{
			// calculate time based on distances and speed
			//  note that calcEncoder() should have set curtbl...
			//NOTE: need to implement correctly!!!
			// to use real distance rather than time
			runtime = a+d;
			// add 25%+1 slop
			runtime += ((runtime >> 2) + 1);
		}
		
		// clear timeout indicator
		motorstat &= ~MStmo;
		
		// start our tick counter in case we need it later
		cntlp->ticks = 0;
		
		// set timeout
		call Timer.start( TIMER_ONE_SHOT, runtime );
	}
	
	/** Startup the motors at specified speed setting.
	 **  reset encoder counters mA,Bclicks = 0
	 **  get PWM settings from curtbl which has
	 **    been set based on given speed setting
	 **  set mA,Bpw globals to values from curtbl entry
	 **      using motorstat bits to select directions
	 **  send PWM values to controller
	**/
	void setPW()
	{
#if  defined(LEDRIGHTLOCK) || defined(LEDLEFTLOCK)
		call Leds.redOff();
#endif
		mAclicks = 0;
		mBclicks = 0;
		
		lAcount = 255;
		lBcount = 255;
	
		// get pulse-width settings based on pre-set direction bits
		mApw = ((motorstat & MSmAcw)) ? curtbl->cwAcnt : curtbl->ccwAcnt;
		mBpw = ((motorstat & MSmBcw)) ? curtbl->cwBcnt : curtbl->ccwBcnt;

		// mark as running
		motorstat |= MSrun;

		// start 'em up
		call Pwm.setABpw8( mApw, mBpw );
		//call Pwm.setABpw8( 90, 90 ); // just testing
	}
	
	/** Start robot moving forward for the given distance and speed.
	 *  @param d -- distance in cm (255 max).
	 *  @param s -- speed in cm/s (255 max).
	 *  @return nada.
	 */
	command void Motors.forward( int16_t d, uint8_t s ) 
	{
		// trap negative values
		if( d < 0 )		d = -d;
		
		// figure out timeout and encoder ticks
		calcEncoder( 0, (uint16_t)d, s );
		calcTimer(   0, (uint16_t)d, s );
		
		// set A=CCW, B=CW, op=move, dir=plus
		motorstat |= (MSmBcw | MSdir);
		motorstat &= ~(MSmAcw | MSturn);
		setPW();
	}
	
	/** backward()
	 *    Start robot moving backward at the given relative speed.
	 *		Will handle negative distances correctly.
	 *  @param d -- distance in cm (255 max).
	 *  @param s -- speed in cm/s (255 max).
	 *  @return nada
	*/
	command void Motors.backward( int16_t d, uint8_t s )
	{
		// trap negative values
		if( d < 0 )		d = -d;
		
		// figure out timeout and encoder ticks
		calcEncoder( 0, (uint16_t)d, s );
		calcTimer(   0, (uint16_t)d, s );
		
		// set A=CW, B=CCW, op=move, dir=minus
		motorstat |= MSmAcw;
		motorstat &= ~(MSmBcw | MSturn | MSdir);
		setPW();
	}
	
	/** rotateCW -- right turn, minus angle --
	 *    Start robot rotating to the right (cw) at the given speed.
	 *		Will handle negative angles correctly.
	 *  @param a -- angle in 1deg increments (180 -- 360deg -- max).
	 *  @param s -- speed in cm/s (255 max).
	 *  @return nada
	*/
	command void Motors.rotateCW( int16_t a, uint8_t s )
	{
		// trap negative values
		if( a < 0 )		a = -a;
		
		// figure out timeout and encoder ticks
		calcEncoder( (uint16_t)a, 0, s );
		calcTimer(   (uint16_t)a, 0, s );
		
		// set A=CCW, B=CCW, op=turn, dir=minus
		motorstat |= MSturn;
		motorstat &= ~(MSmAcw | MSmBcw | MSdir);
		setPW();
	}
	
	/** rotateCCW  -- left turn, plus angle --
	 *    Start robot rotating to the left (ccw) at the given speed.
	 *  @param a -- angle in 1deg increments (180 -- 360deg -- max).
	 *  @param s -- speed in cm/s (255 max).
	 *  @return nada
	*/
	command void Motors.rotateCCW( int16_t a, uint8_t s )
	{
		// trap negative values
		if( a < 0 )		a = -a;
		
		// figure out timeout and encoder ticks
		calcEncoder( (uint16_t)a, 0, s );
		calcTimer(   (uint16_t)a, 0, s );
		
		// set A=CW, B=CW, op=turn, dir=plus
		motorstat |= (MSmAcw | MSmBcw | MSturn | MSdir);
		
		setPW();
	}
	
	/** Jog Left -- adjust MotorA speed (PW set) by given +/- amount.
	 *	 Silently limits at MIN/MAX values.
	 *   @param d -- amount to adjust, plus is always faster.
	 *    Determines actual adjustment based on current motor direction
	 *  @return nada.
	 */
	command void Motors.jogLeft( int8_t d )
	{
		// figure direction and enforce limits
		if( (motorstat & MSmAcw) )	// CW
		{
			// clockwise: smaller number is faster
			mApw += -d;
			if( mApw < MAXCW )
				mApw = MAXCW;
			else if( mApw > MINCW )
				mApw = MINCW;
		}
		else						// CCW
		{
			// CCW: larger number is faster
			mApw += d;
			if( mApw > MAXCCW )
				mApw = MAXCCW;
			else if( mApw < MINCCW )
				mApw = MINCCW;
		}
		
		call Pwm.setApw8( mApw );
	}
	
	/** Jog Right-- adjust MotorB speed (PW set) by given +/- amount.
	 *	 Silently limits at MIN/MAX values.
	 *   @param d -- amount to adjust, plus is always faster.
	 *    Determines actual adjustment based on current motor direction
	 *  @return nada.
	 */
	command void Motors.jogRight( int8_t d )
	{
		// figure direction and enforce limits
		if( (motorstat & MSmBcw) )
		{
			// clockwise: smaller number is faster
			mBpw += -d;
			if( mBpw < MAXCW )
				mBpw = MAXCW;
			else if( mBpw > MINCW )
				mBpw = MINCW;
		}
		else						// CCW
		{
			// CCW: larger number is faster
			mBpw += d;
			if( mBpw > MAXCCW )
				mBpw = MAXCCW;
			else if( mBpw < MINCCW )
				mBpw = MINCCW;
		}
		
		call Pwm.setBpw8( mBpw );
	}
	
	/** enableSpdCntl()
	*    En/Dis-able encoder driven speed control in check*Spd() methods.
	*	TRUE enables, FALSE disables.
	*	Used to take motor control during Target Tracking.
	*  @return nada.
	*/
	command void Motors.enableSpdCntl( bool en )
	{
		if( en )
			motorstat |= MSspdEn;
		else
			motorstat &= ~MSspdEn;
	}
	
	/** getDistance()
	*  @return the distance traveled in CM during this motion.
	*		Note: this is the sum of the click counts from both wheels.
	*		 It will be signed as to fwd/bkd, even during rotations.
	*		Note: encoder value is ~2*CM traveled, it's really just the
	*		 number of encoder clicks that are reported when using
	*		 check{A,B}Spd() and clicks happen about 2/CM.
	*/
	command int16_t Motors.getDistance()
	{
		int16_t diff;
		
		if( motorstat & MStmo )
		{
			// we timed out, estimate distance from time
			// can't tell if we combined move/turn
			// so assume we did one or the other...
			if( motorstat & MSturn )
			{
				return 0;
			}
			
			// use ticks directly
			diff = cntlp->ticks;
			
			// if going backward, make it minus
			if( !(motorstat & MSdir) )
				diff = -diff;
			
			// for zero speed calibration just return time...
			// otherwise calculate
			if( curtbl->spdset == 0 )
			{
				return (diff / BMS);
			}
			else
			{
				return (diff * curtbl->spdset) / BMS;
			}
		}
		else
		{
			// use the actual encoder distances
			if( motorstat & MSturn )
			{
				// if we were turning, the travel distance is
				//  the left-over difference in encoder clicks
				diff = mBclicks - mAclicks;
				
				return (int16_t) (diff * MANGLEFP16) / 16;
			}
			else
			{
/*** could do sanity check...
			// do sanity check
			if( (runtime > (cntlp->ticks + (cntlp->ticks >> 2))) ||
			    (runtime < (cntlp->ticks - (cntlp->ticks >> 2))) )
			{
				// just default to run time
				runtime =  cntlp->ticks;
			}
****/
				// otherwise, moving is the sum of click counters
				diff = mAclicks + mBclicks;
				
				// if going backward, make it minus
				if( !(motorstat & MSdir) )
					diff = -diff;
			
				return diff / (2*PC);
			}
		}
	}
	
	/** getAngle()
	 *  @return the angle traveled in degrees during this motion.
	 *		may be negative if it is a residual difference
     *		when doing a fwd/bkd move, otherwise positive
     *		no matter which direction
	 *		Note: this is the sum of the click counts from both wheels.
	 *		 It will always be positive, even during rotations.
	 *		Note: encoder value is ~2*CM traveled, it's really just the
	 *		 number of encoder clicks that are reported when using
	 *		 check{A,B}Spd() and clicks happen about 2/CM.
	 */
	command int16_t Motors.getAngle()
	{
		int16_t diff;
		
		// NOTE: need to implement correctly:
		
		if( motorstat & MStmo )
		{
			// we timed out, estimate angle from time
			
			// can't tell if we combined move/turn
			// so assume we did one or the other...
			if( !(motorstat & MSturn) )
			{
				return 0;
			}
			
			// use ticks directly
			diff = cntlp->ticks;
			
			// if going backward, make it minus
			if( !(motorstat & MSdir) )
				diff = -diff;
			
			// for zero speed calibration just return 0...
			// otherwise calculate
			if( curtbl->spdset == 0 )
			{
				return 0;
			}
			else
			{
				// NOTE: fix...
				// get CM traveled,
				// multiply by fix-point val,
				// divide by 8 (since encoders give 2x CM travel)
				// and do it all such that 16 bit calcs work ok...
				return  (((diff * curtbl->spdset) / (BMS/8)) *
								  MANGLEFP16) /
								   (4*16);
			}
		}
		else
		{
			// use the actual encoder distances
			if( !(motorstat & MSturn) )
			{
				// if we were moving, the angle change is
				//  the left-over difference in encoder clicks
				diff = mBclicks - mAclicks;
				
				// return will be minus if relative CW 
				return (diff * MANGLEFP16) / 16;
			}
			else
			{
				// sum it up
				diff = mAclicks + mBclicks;
				
				// if going CW, make it minus
				if( !(motorstat & MSdir) )
					diff = -diff;
			
				// otherwise, turning is the average
				// do angle calc in fix-point*16
				return (diff * MANGLEFP16) / (16*2);
			}
		}
	}
	
	/** getSpeed()
	*  @return the actual travel speed 
	*  NOTE: FIX  and check
	*/
	command uint8_t Motors.getSpeed()
	{
		return (uint8_t) curtbl->spdset;
	}
	
	/** stopMotors()
	 *   Stop both the PWM outputs and set outputs to 0.
	 *	 Stop Timer just in case...
	 *	 Save motor PWM values for future use.
	 *	 Calculate and update Completion distances in cntlp.
	 *	 Notify User via MotorStatus.statusReady() if sig==TRUE
	 *  @return nada.
	 */
	command void Motors.stopMotors( bool sig )
	{
		// stop motors, timer, encoders
		call Pwm.stopAll();
		call Timer.stop();
		call EncoderStatus.doStatus( 0 );
		motorstat &= ~MSrun;
		
		// tell em
		// user should call getStatus() for real results in cntlp
		if( sig )
			signal MotorStatus.statusReady( motorstat );
	}
		
/********* these do speed control on the actual speed **********
	/** checkASpeed()
	*   Check the Left--MotorA--speed against given count value.
	*	Adjust MotorA pulse width if necessary
	*	 and save new pw setting for use later in speed table.
	*   Only called when encoder speed control is enabled.
	*  @return nada
	*
	void checkASpeed( uint8_t count )
	{
		// adjust motor if enabled and we are over the accel phase
		if( (mAclicks > ACCELCLICK) )
		{
			// sanity check signal
			if( (count > minEticks) && (count < maxEticks) )
			{
#ifdef LEDLEFTLOCK
				call Leds.redOff();
#endif
				// got a tick count in range
				if( count < minTWin )			// going too fast
					call Motors.jogLeft( -JOGINCR );	// slow down
				else if( count > maxTWin )
					call Motors.jogLeft( JOGINCR );		// speed up
				else							// save good PW value
				{
					if( (motorstat & MSmAcw) )
						curtbl->cwAcnt = mApw;
					else
						curtbl->ccwAcnt = mApw;
#ifdef LEDLEFTLOCK
					call Leds.redOn();
#endif
				}
			}
			else
			{
				// negate the click counters and forget the encoders
				mAclicks = 0;
				motorstat &= ~MSspdEn;
			}
		}
	}
	
	/** checkBSpeed()
	*   Check the Right--MotorB--speed against given count value.
	*	Adjust MotorB pulse width if necessary
	*	 and save new pw setting for use later in speed table.
	*   Only called when encoder speed control is enabled.
	*  @return nada
	*
	void checkBSpeed( uint8_t count )
	{
		// if acceleration period over...
		if( (mBclicks > ACCELCLICK) )
		{
			// sanity check signal
			if( (count > minEticks) && (count < maxEticks) )
			{
#ifdef LEDRIGHTLOCK
				call Leds.redOff();
#endif
				if( count < minTWin )			// running fast
					call Motors.jogRight( -JOGINCR );	// slow down
				else if( count > maxTWin )
					call Motors.jogRight( JOGINCR );	// speed up
				else							// save good PW value
				{
					if( (motorstat & MSmBcw) )
						curtbl->cwBcnt = mBpw;
					else
						curtbl->ccwBcnt = mBpw;
#ifdef LEDRIGHTLOCK
					call Leds.redOn();
#endif
				}
			}
			else
			{
				// negate the click and forget the encoders
				mBclicks = 0;
				motorstat &= ~MSspdEn;
			}
		}
	}
********************* end actual speed control ************/
	
/********** these do speed control on the acceleration *****/
	/** checkASpeed()
	*   Check the Left--MotorA--speed against given count value.
	*	Adjust MotorA pulse width if necessary
	*	 and save new pw setting for use later in speed table.
	*   Only called when encoder speed control is enabled.
	*  @return nada
	*/
	void checkASpeed( uint8_t count )
	{
#ifdef LEDLEFTLOCK
		call Leds.redOff();
#endif
		// if acceleration period over
		//   note that this doesn't matter
		//   also need to do sanity Error window check...?
		//if( (mAclicks > ACCELCLICK) )
		{
			// note: count is small when going fast...
			// do nada if below the desired speed and speeding up,
			// otherwise give a plus poke
			// OR
			// do nada if above the desired speed and slowing down,
			// otherwise give a minus poke
			// OR
			// we seem to be at the right speed, save data
			if( count > maxTWin )		// going too slow
			{
				if( count >= lAcount )	// and slower/same as before
				{
					// speed up a bit
					call Motors.jogLeft( JOGINCR );	
				}
			}
			else if( count < minTWin )	// going too fast
			{
				if( count <= lAcount )	// and faster/same as before
				{
					// slow down up a bit
					call Motors.jogLeft( -JOGINCR );
				}
			}
			else						// just right
			{
				// save the setting
				if( (motorstat & MSmAcw) )
					curtbl->cwAcnt = mApw;
				else
					curtbl->ccwAcnt = mApw;
#ifdef LEDLEFTLOCK
				call Leds.redOn();
#endif
			}
			
			// save for next time
			lAcount = count;
		}
		
		return;
	}
	
	/** checkBSpeed()
	*   Check the Right--MotorB--speed against given count value.
	*	Adjust MotorB pulse width if necessary
	*	 and save new pw setting for use later in speed table.
	*   Only called when encoder speed control is enabled.
	*  @return nada
	*/
	void checkBSpeed( uint8_t count )
	{
#ifdef LEDRIGHTLOCK
		call Leds.redOff();
#endif
		// if acceleration period over
		//   note that this doesn't matter
		//   also need to do sanity Error window check...?
		//if( (mBclicks > ACCELCLICK) )
		{
			// note: count is small when going fast...
			// do nada if below the desired speed and speeding up,
			// otherwise give a plus poke
			// OR
			// do nada if above the desired speed and slowing down,
			// otherwise give a minus poke
			// OR
			// we seem to be at the right speed, save data
			if( count > maxTWin )		// going too slow
			{
				if( count >= lBcount )	// and slower/same as before
				{
					// speed up a bit
					call Motors.jogRight( JOGINCR );	
				}
			}
			else if( count < minTWin )	// going too fast
			{
				if( count <= lBcount )	// and faster/same as before
				{
					// slow down up a bit
					call Motors.jogRight( -JOGINCR );
				}
			}
			else						// just right
			{
				// save the setting
				if( (motorstat & MSmBcw) )
					curtbl->cwBcnt = mBpw;
				else
					curtbl->ccwBcnt = mBpw;
#ifdef LEDRIGHTLOCK
				call Leds.redOn();
#endif
			}
			
			// save for next time
			lBcount = count;
		}
		
		return;
	}
	
	/** Dummy function for this module.
	 */
	command result_t MotorControl.dACycle() { return SUCCESS; }
	  
	/** Get system status.
	 *  Calculate distance and angle traveled and set cntlp fields.
	 *  @return bitmask of interesting status
	 */
	command uint8_t MotorControl.getStatus()
	{
		// calculate and update distances
		cntlp->distance = call Motors.getDistance();
		cntlp->angle = call Motors.getAngle();
		cntlp->speed = call Motors.getSpeed();
		
#ifdef SENDENCVAL
		// send the encoder counts for debugging poiposes
	    statp->pingA1cnt = mAclicks;
	    statp->pingB1cnt = mBclicks;
#endif
		
		return motorstat;
	}
		
/**** Completion signals ****/

	/** what to do when our Motor Timer fires
	 ** Probably timed out on failed encoder...
	**/
	event result_t Timer.fired()
	{
		//call Leds.redToggle();
		
		// set timeout indicator
		motorstat |= MStmo;
		
		call Motors.stopMotors( 1 );
		
		return SUCCESS;
	}
	
	/** Event sent when Encoder tick is ready,
	 **  Increment {mA,mB}clicks counters to check how far we went.
	 **  Adjust motor speeds as needed.
	 ** note: if the encoders go crazy-pulsing the click counts
	 **  will also go crazy and make the robot stop way too soon...
	 **    
	 ** @param encstat -- interesting bits of information
	 **  					see BESTATUS enum for details.
	 **   Actual encoder times are in cntlp->enc{A,B}ticks
	 **/
	async event result_t EncoderStatus.statusReady( uint8_t encstat )
	{
		// if we are running count clicks
		if( motorstat & MSrun )
		{
			if( encstat & BEeAdata )
			{
				++mAclicks;
				// if under speed control, do the thing
				if( motorstat & MSspdEn )
					checkASpeed( cntlp->encAticks );
			}
				
			if( encstat & BEeBdata )
			{
				++mBclicks;
				// if under speed control, do the thing
				if( motorstat & MSspdEn )
					checkBSpeed( cntlp->encBticks );
			}
		}
		
		// stop if we are done
		if( (mAclicks + mBclicks) >= enClicks )
			call Motors.stopMotors( 1 );
	
		return SUCCESS;
	}
	
	
/****	MotorStatus interface -- for user notification ****/

	/** Sign up for or cancel status ready signaling.
	 *  Could shut off stuff, but we probably don't care...
	 *  Result status is provided by signal of statusReady().
	**/
	command result_t MotorStatus.doStatus( uint8_t data )
	{
		return SUCCESS;
	}
  
	/** Event sent when data acquisition cycle is completed,
	 **    User should implement to get Motor completion event.
	 ** @param status -- interesting status bits
	 **  see various status enums for values
	**/
	default async event result_t MotorStatus.statusReady( uint8_t status )
		{ return SUCCESS; }
		
} // end'o'HSMotors impl
