/** BumpEncM.nc -- Implementation of RoboCar
 **			 Bumper and wheel Encoder functions.
 **		Init() function sets up PortC input for all users.
 **		Gets called regularly by an external timer.
 **		Reads and processes PortC value.
 **		Parses the results and hands them to the Motion control code.
 **		Users may use the SensorStatus interfaces to "subscribe"
 **		 to a signal-callback function to get updates as they happen.
**/

includes RoboMsg;  /* the basic request and status messages */


module BumpEncM
{
 	provides interface SensorControl;
 	provides interface SensorStatus as BumperStatus;
 	provides interface SensorStatus as EncoderStatus;
 	uses
	{
 		interface Leds;
 		interface HPLPort as PortData;
	}
}
implementation
{
	// point to 'globals' shared by main modules, so syntax is the same
	RoboStatusMsg *statp;	// outgoing status message body
	RoboControl *cntlp;		// ongoing shared control data
	
	uint8_t bestat = 0;		// our ongoing status, bumpers and encoders
	
	/**
	* Used to initialize this component.
	*/
	command result_t SensorControl.init()
	{
		// crank up the PORT C input
		call PortData.init( 0 );
		
		return SUCCESS;
	}
	
	/** Probably superfluous **/
	command result_t SensorControl.start() { return SUCCESS; }
	command result_t SensorControl.stop() { return SUCCESS; }
	
	/** set the global status message pointers --
	 **		 need status body and control buffer
	**/
	command result_t SensorControl.setGlobals(
							 TOS_Msg *statmsg,		// status return
  							 RoboCmdMsg *cmdbuf,	// current command
  							 RoboControl *cntlbuf ) // internal data
	{
		statp = (RoboStatusMsg*)statmsg->data;
		cntlp = cntlbuf;
		return SUCCESS;
	}
	
	/****
		Read the input PortC on every timer tick.
		Detect first ON state of each encoder bit
		 and call Motor encoder code with time-count.
		 Counts are passed through cntlp->enc{A,B}ticks.
		Detect change of ON/OFF bumper state and
		 accumulate bumper signals until we have been
		 called for BUMPACCUM ticks (approx 20 times/sec).
		 Then call Motion bumper code with the new bumper data.
		 Bumper data is passed through cntlp->bumpdata.
		 After the BUMPACCUM period keep updating bumpdata,
		  but don't call Motion code with changes until
		  it cycles ON or OFF.
	****/
	
	/** Process encoder data from input switches.
	 * Always called from timer.fired() event.
	 * Post results to cntlp struct and notify user if requested.
	 * @param port -- rectified PortC data.
	 */
	// accumulators for encoder signals
	uint8_t eAticks = 0;	// encoder A tick count
	uint8_t eBticks = 0;	// encoder B tick count
	
	void encoderDetect( uint8_t port )
	{
		// deal with wheel encoder clicks on both sides
		++eAticks;	// encoder A tick count
		++eBticks;	// encoder B tick count
		if( (port & WENCA) == 0 )
		{
			// save current state == 0
			bestat &= ~(BEeAstate);
		}
		else if( !(bestat & BEeAstate) ) // first sample that is 1
		{
			// save current state == 1, set new data flag
			bestat |= (BEeAstate | BEeAdata);
			// get the time
			cntlp->encAticks = eAticks;
			eAticks = 0;
			// tell the user if they want to know
			if( bestat & BEesignal )
				signal EncoderStatus.statusReady( bestat );
			// reset new data flag
			bestat &= ~(BEeAdata);
		}

		if( (port & WENCB) == 0 )
		{
			// save current state
			bestat &= ~(BEeBstate);
		}
		else if( !(bestat & BEeBstate) ) // first sample that is 1
		{
			// save current state, set new data flag
			bestat |= (BEeBstate | BEeBdata);
			// get the time
			cntlp->encBticks = eBticks;
			eBticks = 0;
			// tell the user if they want to know
			if( bestat & BEesignal )
				signal EncoderStatus.statusReady( bestat );
			// reset new data flag
			bestat &= ~(BEeBdata);
		}
		
		return;
	}

	/** Process data from PortC input switches.
	 * Always called from timer.fired() event.
	 * Accumulate bumper results.
	 * Note that all signals are assumed to be active HIGH,
	 *  so RoboCar Bumper bits should be inverted prior to call.
	 * Deals with the bumper holdoff bumpdelay and forces a user
	 *  signal if the bumpers don't change or if they clear.
	 * @param port -- rectified PortC data (inverted and masked).
	 */
	// accumulators for bumper signals
	uint8_t b1cnt = 0;		// bumper 1 bit0
	uint8_t b2cnt = 0;		// bumper 2 bit1
	uint8_t b3cnt = 0;		// bumper 3 bit4
	uint8_t b4cnt = 0;		// bumper 4 bit8
	uint8_t cbcnt = 0;		// collision detector bit5
	uint8_t lastbump = 0;	// save last accumulated bumper data
	int8_t bumpticks = -1;	// bumper tick count, flag off with -1
	int16_t bumpdelay = -1;	// bumper holdoff, flag off with -1
	
	// number of ticks to accumulate new bumper data
	#define BUMPACCUM 50
	// factor to calculate bumpdelay count, ~250 for 25cm/sec speed
	#define BUMPDISTANCE 6500
	
	void bumperAccum( uint8_t port )
	{
		// deal with bumper holdoff delay 
		//  (only when starting with bumpers already active)
		// note: after delay completes we want
		//   to force a new signal accumulation
		if( bumpdelay > 0 )					// doing delay
		{
			--bumpdelay;
		}
		
		// accumulate signals constantly
		if( bumpticks > 0 )
		{
			// will trigger update behavior when == 0
			--bumpticks;
			
			// accumulate each bumper bit separately
			//  note: all bits are ON = 1, after fiddling
			if( (port & 0x01) )
				++b1cnt;
			if( (port & 0x02) )
				++b2cnt;
			if( (port & 0x04) )
				++b3cnt;
			if( (port & 0x08) )
				++b4cnt;
			if( (port & 0x10) )
				++cbcnt;
		}
		else
		{
			// reset the accumulators			
		    // and start over with accumulation timer
			b1cnt = b2cnt = b3cnt = b4cnt = cbcnt = 0;
		    bumpticks = BUMPACCUM;
		}
			
		return;
	}
			
	/** Analyzes bumper results, stuffs cntlp->bumpdata,
	 *   and signals user as needed.
	 *   if accumulating a new signal just wait,
	 *    otherwise update bumpdata on each pass.
	 * @param port -- rectified PortC data.
	 */
	void bumperDetect( uint8_t port )
	{
		uint8_t newdata = 0;
		
		// when our accum ticks underflow see if we need to report
		// if the accumulated number of ON bits is above the threshold,
		//  then set the appropriate bumper bits
		if( bumpticks == 0 )
		{
			if( b1cnt > (BUMPACCUM/4) )
				newdata |= 0x01;
			if( b2cnt > (BUMPACCUM/4) )
				newdata |= 0x02;
			// Note: the schmick who laid out the bumper board reversed...
			//   or didn't...the rear signals, so re-reverse them...
			if( b3cnt > (BUMPACCUM/4) )
				newdata |= 0x08;
			if( b4cnt > (BUMPACCUM/4) )
				newdata |= 0x04;
			if( cbcnt > (BUMPACCUM/4) )
				newdata |= 0x10;
				
			// post accumulated bumper data for user Status report
			// and flag if we have a clear bumper signal at any time
			cntlp->bumpdata = newdata;
			if( newdata == 0 )
				cntlp->monstate |= RMS_BUMPCLR;
			
			// see if anything changed
			if( newdata != lastbump )
			{
				// set new data flag
				//  note: BEbstate ON/OFF is already set correctly
				bestat |= BEbdata;
				lastbump = newdata;
			}
		}
		
		// if we have a new accumulated signal, or
		// if a bump holdoff delay times-out
		// do housekeeping and signal the user
		if( (bestat & BEbdata) || (bumpdelay == 0) )
		{
			// if doing holdoff
			if( bumpdelay >= 0 )
			{
				// if holdoff expired without clearing, make a report
				if( bumpdelay == 0 )
				{
					// force new data flag
					bestat |= BEbdata;
				}
				else if( lastbump != 0 )
				{
					// still waiting for bumper clear
					// just skip it
					bestat &= ~BEbdata;
					return;
				}
			
				// don't do it again, til next time
				bumpdelay = -1;
			}
				
			//  call user if they want us
			if( bestat & BEbsignal )
			{
				signal BumperStatus.statusReady( bestat );
			}
			
			// reset new data flag so we don't get confused
			bestat &= ~(BEbdata);
		}
			
		return;
	}
	
	/** Initiate a conversion sequence.
	 *  Called by external timer function,
	 *  Results are posted into the global RoboControl struct
	 *  Users can get synchronized results using the SensorStatus I/F.
	 */
	command result_t SensorControl.dACycle()
	{
		// get data
		uint8_t port = call PortData.read();
		// deal with encoder signals
		encoderDetect( port & ~CRASHBITS );
		// invert bumpers because they are active low
		port ^= BUMPERS;
		// wipe out stuff we don't care about
		port &= CRASHBITS;
		// accumulate bumper info
		bumperAccum( port );
		// deal with bumper signals
		bumperDetect( port );
		
		return SUCCESS;	
	}
	
	/** Check on system status, immediate return
	 **  copy current bumper data into status message
	 **  leave it to the motor code to report encoder data...
	 ** @return bitmask of interesting status
	 **  see BESTATUS enum for values
	 **/
	command uint8_t SensorControl.getStatus()
	{
		// copy current bumper data into status message
		statp->bumpers = cntlp->bumpdata;

		return bestat;
	}
	
	/** Sign up for or cancel signaling on Bumper status.
	 *  If "hold" is != 0 the user's signal routine will be
	 *   called for each bumper change, 0 cancels calling.
	 *	If hold is 0xff no bumper hold-off delay will be initiated
	 *   otherwise the hold value will be taken to be a robot speed
	 *   and used to calculate the number of sample ticks to wait
	 *   before reporting bumper results (in the hopes that an
	 *   active bumper signal will clear after some small move).
	 * 
	 *  Result status is provided in statusReady().
	**/
	command result_t BumperStatus.doStatus( uint8_t hold )
	{
		if( hold )
		{
			// report changes to user
			bestat |= BEbsignal;
			
			// do bumper holdoff if requested
			if( hold != 0xff )
		 		bumpdelay = BUMPDISTANCE / hold;
		}
		else
		{
			// stop reporting
			bestat &= ~BEbsignal;
		}
			
		return SUCCESS;
	}
		
	/** Event sent when Bumper status is ready,
	 **    user should implement.
	 ** @param bumpstat -- interesting bits of information
	 **  see BESTATUS enum for details. Actual bumper data
	 **  is in cntlp->bumpdata
	 **/
	default async event result_t
		BumperStatus.statusReady( uint8_t bumpstat )
		{ return SUCCESS; }
		
	/** Sign up for or cancel signaling on Encoder status.
	 *  If "yes" is TRUE user's signal routine will be called
	 *   after each conversion cycle, FALSE cancels calling.
	 *  Result status is provided in statusReady().
	**/
	command result_t EncoderStatus.doStatus( uint8_t yes )
	{
		if( yes )
			bestat |= BEesignal;
		else
			bestat &= ~BEesignal;
			
		return SUCCESS;
	}
		
	/** Event sent when Encoder clicks are received,
	 **    user should implement.
	 ** @param status -- interesting status bits
	 **  see ENCSTATUS enum for values
	**/
	default async event result_t
		EncoderStatus.statusReady( uint8_t status )
		{ return SUCCESS; }
		
} //end'o'BumpEncM impl
