/** RoboMessage.java -- basic RoboCar message structure.
**/
package com.etantdonnes.tinyos.robo;

import java.io.PrintStream;


/** Base class for RoboII communications using basic Mica2 message structure
**/
public class RoboMessage
{
	// sizes
	public static final int MSGSIZE = 34;	// bytes in entire message
	public static final int PKTSIZE = 29;	// payload bytes in packet

//	public static final int DESTADDR = 0x0001;	// individual Mote
	public static final int DESTADDR = 0xffff;	// broadcast
	public static final int GROUPID  = 0x33;	// default group in Makelocal
	public static final int SOURCEID  = 0;		// my default TOSBase ID

	// types -- from RoboMessage.h
    public static final int AM_ROBOSTATUSMSG = 10;	// Status (Send to Host)
    // Host Commands (Receive from Host)
    public static final int AM_ROBOCMDSTARTMSG = 32; // Start motion
    public static final int AM_ROBOCMDSTOPMSG = 33;	// Stop immediately
    public static final int AM_ROBOCMDSTATMSG = 34;	// Status: return current 
	
	// message payload packet buffer (alloc in sub-classes)
	protected byte[] data = null;

	// components of the standard Mica2 message
	public int destAddr;		// destination MoteID ( and RoboII device ID)
	public int handlerID;		// message ID
	public int groupID;			// mote group ID (arbitrary, assigned in TOS make)
	public int msgLen;			// payload size (always PKTSIZE for this...)

	// common components of the RoboII messages
	public int sourceMoteID;	// Sender's MoteID
    public int sequenceNumber = 0; // Message sequence number

    
	public RoboMessage()
	{
	}

	/** initialize the basic fields
	 **/
	public RoboMessage( int da, int hid, int gid, int len, int sid )
	{
		this.destAddr  = da;
		this.handlerID = hid;
		this.groupID   = gid;
		this.msgLen    = len;
		this.sourceMoteID = sid;
	}

	/** Parse a message packet into our local fields.
	 *  @param -- byte array containing data read from Mica2.
	 *  @return -- true if OK,
	 *  		   false if something wrong with packet.
	 *
	 * NOTE: all the offsets and sizes are hard coded here...
	 *   and for now, all packets and message lengths are the same.
	 **/
	public boolean parse( byte[] p )
	{
		boolean rval = false;

		// check the buffer and message lengths just in case
		if( (p.length == MSGSIZE) &&			// buffer size
			((p[4] & 0xff) == PKTSIZE) )	// RoboStatMsg length
		{
			// get 'standard' Mica2 message header
			destAddr  = ti(p, 0) & 0xffff;
			handlerID =    p[2]  & 0xff;
			groupID   =    p[3]  & 0xff;
			msgLen    =    p[4]  & 0xff;

			// RoboStatusMsg, as defined in TOS-Robot code RoboMsg.h
			sourceMoteID	= ti(p, 5) & 0xffff;
    		sequenceNumber	= ti(p, 7) & 0xffff;

			rval = true;	// A-OK
		}

		return rval;
	}

	/** set common fields in given data buffer
	 **/
	protected void setFields( byte[] p )
	{
		fi( destAddr, p, 0 );
		p[2] = (byte) (handlerID & 0xff);
		p[3] = (byte) (groupID   & 0xff);
		p[4] = (byte) (msgLen    & 0xff);
		fi( sourceMoteID, p, 5);
		fi( ++sequenceNumber, p, 7);	// command receive ignores non-incr #'s

		return;
	}

	/** set common fields in local data buffer and 
	 * @return the full message buffer
	 **/
	public byte[] dataGet()
	{
		if( data != null )	setFields( data );
		return data;
	}

	/** set all the local fields to 0 
	 * @return nada
	 **/
	public void clearPacket()
	{
		if( data != null )
			for( int i=0; i <data.length; ++i )		data[i] = 0;
		return;
	}


	/** convert local values to readable format
	 **/
	public String toString()
	{
		return(
			// 'standard' Mica2 message header
				"d=" + destAddr +
				" h=" + handlerID +
				" g=" + groupID +
				" l=" + msgLen +
			// common fields, as defined in RoboMessage.h
				" s=" + sourceMoteID +
				" n=" + sequenceNumber +
				" ::\t"
			  );
	}

	// print given message data as bytes
	public static void printBytes( PrintStream out, byte[] p, String hdr )
	{
		System.out.print(   hdr + "(" + p.length + ")" +
					 " d=" + (ti(p, 0) & 0xffff) +
					 " h=" + (p[2]  & 0xff) +
					 " g=" + (p[3]  & 0xff) +
					 " l=" + (p[4]  & 0xff) +
					 " s=" + (ti(p, 5) & 0xffff) +
					 " n=" + (ti(p, 7) & 0xffff) +
					 " ::\t"
				  );
			  
		for( int i=9; i < p.length; ++i )
			System.out.print( " " +  (p[i] & 0xff) );
		
		System.out.println("");
	}

	// FromInt -- Java to TOS
	//  convert from a Java int
	//  into a little-*f*ck*ng*-endian int in the buffer at byte 'start'
	protected static void fi( int data, byte[] buf, int start )
	{
		buf[start]   = (byte) (data & 0xff);
		buf[start+1] = (byte) ((data >> 8) & 0xff);
	}

	// ToUInt -- TOS to Java
	// convert buffer at byte 'start'
	//  from a little-*f*ck*ng*-endian int into a Java int, without sign
	protected static int ti( byte[] buf, int start )
	{
		return ((buf[start+1] << 8) & 0xff00) + (buf[start] & 0xff);
	}

	// ToSignedInt -- TOS to Java
	// convert 2-byte buffer at byte 'start'
	//  from a little-*f*ck*ng*-endian int into a signed Java int
	// sign extend if necessary
	protected static int tim( byte[] buf, int start )
	{
		int x = ((buf[start+1] << 8) & 0xff00) + (buf[start] & 0xff);
		if( (x & 0x8000) != 0 )
		{
			// they were trying to tell us the value is negative...
			// so extend the sign to 4 bytes
			x = (x | 0xffff0000);
		}
		return x;
	}

} // end'o'RoboMessage