/** ReadThread.java -- Independent thread for reading from Mote
 *   Assumes that it is going to receive a RoboStatMsg.
 *   Logs message contents as per various test states.
 * 
 *  Taken loosly from Listen.java,v 1.3.2.3 2003/08/20
**/
package com.etantdonnes.tinyos.robo;

import net.tinyos.packet.PacketSource;


public class ReadThread extends Thread
{
	protected boolean runme = false;	// in order to kill thread cleanly
	protected Thread threadID;			// thread ID for housekeeping
	protected PacketSource roboComm;	// TOS rd/wr object
	protected LogMessage log = null;	// logging scheme to use
	public RoboStatMsg msg = new RoboStatMsg(); // latest received data

	/** Make a new thread to read from Mote
	 * 
	 * @param rc -- communication object
	 * @param l -- initial logging mechanism
	 */
	public ReadThread( PacketSource rc, LogMessage l )
	{
		runme = true;	// set to false to exit thread
		roboComm = rc;
		log = l;
		
        // start the read thread
        threadID = new Thread( this, "ReadThread" );
        threadID.start();
	}
	
	/** set the logging style to use when messages are received
	 *    probably should be synced somehow so we don't accidentally
	 *     wipe out the logger while it is being used in the read
	 *      thread itself...but there are problems with write/read
	 *      access when the message timeing is really short.
	 * 
	 * @param l -- new logging object
	 */
	public void setLogging( LogMessage l )
	{
		log = l;
	}

    /** terminate with extreme prejudice
     *   force run() to break as best we can
     */
    public void kill()
    {
        runme = false;

        // neither of interrupt or stop seem to work on NT....
        // a blocked Stdin.read() just never breaks w/o calling exit.
        // but we try them anyway...
        threadID.interrupt();  // does this work?..apparently....
    }

	/** run the read thread
	 *   read mote message,
	 *   parse message
	 *   log results using logger specified above.
	 *  bails out on errors or when runme is falsified
	 */
    public void run()
    {
		byte[] packet;
		try
		{
			while( runme && ( (packet = roboComm.readPacket()) != null ) )
			{
				msg.parse( packet );
				
				// NOTE: this may still be a race when the log
				//   is replaced, but it's just a test program...
				if( log != null )
					synchronized( log )
					{
						log.logRead( msg );
					}
			}

		} catch( Exception e )
		{
			System.err.println( "ReadThread exception " + roboComm.getName() +
						": " + e.getMessage() );
		}

		System.err.println( "Read Thread bailing out, mon." );
		return;
	}

} // end'o'ReadThread