/** LogMessageTest.java -- Logging routines for CommTest loop.
 *  Assumes that you create a new one everytime you run a test loop.
**/
package com.etantdonnes.tinyos.robo;

import java.io.PrintStream;
import java.text.NumberFormat;
import java.util.Vector;


public class LogMessageTest implements LogMessage
{
	protected long starttime = 0L;	    // tick count when starting auto test
	protected PrintStream out = null;	// output stream
	
	// place to save write messages until reply is received
	protected Vector writes = null;
	protected int devs[] = null;	// list of devices we want to hear from
	protected int numdevs = 0;	// number of devs in list
	protected logaccum las[] = null; // accumulators for each device's results
	
	protected boolean bcast = false;// true if broadcast test, false for R-R
	// these are for logging test parameters
	protected int autodelay = 0;	// message transmit delay
	protected int runstate = 0;		// statReq runstate indicators
	protected int txstrength = 0;	// statReq transmit strength (0 == full)
	
	// used by logaccum inner class
	boolean didhdr = false;	// print accum header only once...
	logaccum aggAccum = null;	// aggregate collector

	/** make a new loop logger
	 * 
	 * @param o -- output stream to write log
	 * @param d -- array of devices from which we expect to get answers
	 * @param n -- number of entries in the array
	 * @param b -- this is a broadcast test if true, round robin if false
	 *              if bcast the first entry in the device array will be -1...
	 * @param ad -- send message delay time
	 * @param rs -- statReq runstate params
	 * @param tx -- reMote transmit power setting
	 */
	public LogMessageTest( PrintStream o, int[] d, int n,
							int ad, int rs, int tx )
	{
		out = o;
		devs = d;
		numdevs = n;
		bcast = (devs[0] == -1);
		// these are just for logging test parameters
		autodelay = ad;
		runstate = rs;
		txstrength = tx;
		
/** need to make this a lookup list....
 *  assumes that devs are sequential and start from 1...
 */
		// init accumulators .. just make 10 of em...
		las = new logaccum[10];
		for( int i=0; i<las.length; ++i )
			las[i] = new logaccum( bcast, numdevs );
	
		aggAccum = new logaccum( bcast, numdevs );
		
		// init message write save
		writes = new Vector(20);
		
		// put header on data
		out.println(  "\ndev\tts\tsn\twr\tmsg\t!ack\trssi\tna\tnm" );
	}
	
	public void startLog( long start )
	{
		starttime = start;
	}

	public void stopLog()
	{
		starttime = 0L;
		
		// see if anything is missing still
		flushDropped( (int)(System.currentTimeMillis()-starttime), -1 );
		
		// print accumulators
		for( int i=0; i<las.length; ++i )
			out.println( las[i].makeString() );
		
		// print aggregate accumulator
		out.println( aggAccum.makeString() );
		
		// wipe out everything
		writes = null;
		las = null;
		devs = null;
	}
	
	/** log a message write (Status Request)
	 * 
	 *   This behaves differently depending if we are doing
	 *    a bcast message or a round-robin set. See comments below.
	 * 
	 * @param start -- CurrentTime when we started writing
	 * @param acked -- was the message ACKed?
	 * @param statReq -- The request object we used
	 */
	public void logWrite( long now, boolean acked, RoboStatReq statReq )
	{
		int dai = 0;	// destAddr array index
		int devID = 0;
		
		// save write data for later when we hope to get a reply
		// make one copy of the record for each device we expect to hear from
		// if we did a bcast message (devs[0]==-1) then generate a whole bunch
		if( bcast )
			dai = 1;	// skip first -1 bcast id and get list of real ones
		else
			dai = 0;	// only do the one in the statReq itself (R-R test)
			
		do
		{
			if( bcast )
				devID = devs[dai];
			else	
				devID = statReq.destAddr;
			
			// make all the stuff to save
			wmd d =  new wmd( (int)(now - starttime),
					          (int)(System.currentTimeMillis() - now),
					          acked,
					          devID, statReq.sequenceNumber );
			// and save it for later
			writes.add( d );
			
		  // only do the one in the Req (for round-robin test),
		  //  otherwise for bcast, make one entry for each in devs list
		} while( (dai != 0) && (++dai < numdevs) );
						
		
		
		/**
		out.println( start-starttime + "\t" + "W\t" +
				 statReq.sequenceNumber + "\t" +
				 (System.currentTimeMillis() - start) + "\t\t" +
				 (acked ? "0" : "1") );
		**/
	}
	
	public void logRead( RoboStatMsg msg )
	{
		wmd d = null;
		// get current relative time
		int now = (int) (System.currentTimeMillis()-starttime);
		
		// find our seqnum in writes list
		int idx = findSN( msg.sequenceNumber, msg.sourceMoteID );
		if( idx == -1 )		return;		// duplicate or something...
		
		// do various calculations on real msg and print results
		d = (wmd) writes.get( idx );
		writes.remove( idx );
		
		// log a successful message
		logSucc( d, now, msg );
		
		// check to see if any missing replies should be logged
		flushDropped( now, idx );
		
		/**
		out.println(
		 System.currentTimeMillis()-starttime +
		 "\tR\t" +
		 msg.sequenceNumber + "\t\t" +
		 msg.rssi );
		**/
	}

	/** Flush out any stragglers in the write list that have not
	 *   received a reply and log them as Dropped.
	 * 
	 * @param now -- time we started this
	 * @param idx -- last list index we found (work from 0 to idx)
	 * 				  if idx == -1, flush the whole list
	 */
	protected void flushDropped( int now, int idx )
	{
		wmd d;
		
		// see if we want to flush the whole thing
		if( idx == -1 )
		{
			idx = writes.size();
		}
		
		int didx = 0;
		do
		{
			// ok this is a little tricky...
			// We want to look between index 0 and whatever
			//  we found to be our current message index (idx).
			// We've already removed idx, but it's still the
			//  end of the line marker when we start.
			// Then...when we find a dropped guy we get a didx,
			//  which we process and then remove.
			// So...we want to start at the SAME didx because it
			//  now points to what was the _next_ element in the
			//  list, AND we want to decrement the end (idx) because
			//  it has also moved up one.
			// Hopefully this makes sense when I try to read it later...
			if( (didx = findDropped( didx, idx, now )) != -1 )
			{
				// do various calculations and print results
				d = (wmd) writes.get( didx );
				writes.remove( didx );
				
				// log the failures
				logFail( d );
			}
			idx -= 1;
		} while( didx != -1 );
	}

	protected void logSucc( wmd d, int now, RoboStatMsg msg )
	{
		int msgtime = (now - d.start);

		// save it all for later
		las[d.devid].storeSucc( d, msgtime, msg );
	
		// devid time seqnum wrtime cycletime !acked  reply-rssi noacks nomsgs
		out.println( d.devid + "\t" +
				 d.start + "\t" +
				 d.seqnum + "\t" +
				 d.time + "\t" +
				 msgtime + "\t" +
				 (d.acked ? "0" : "1") + "\t" +
				 msg.rssi + "\t" +
				 msg.noacks + "\t" +
				 msg.nomsgs );
	}

	protected void logFail( wmd d )
	{
		// save it all for later
		las[d.devid].storeFail( d );
	
		// devid, timestamp, seqnum, wrtime, drop, !acked, XXX
		out.println( d.devid + "\t" +
				     d.start + "\t" +
					 d.seqnum + "\t" +
					 d.time + "\t" +
					 "Drop" + "\t" +
					 (d.acked ? "0" : "1") + "\t" +
					 "xxx" );
	}

	
	/** find the specified sequence number in writes list
	 * 
	 * @param seqnum
	 * @return index of writes entry, or -1 if failed
	 */
	protected int findSN( int seqnum, int devid )
	{
		int i;
		
		// scrod the iterators, full indexing ahead
		for( i=0; i!=writes.size(); ++i )
			if( ((wmd)writes.get(i)).comp( seqnum, devid) )
				break;
			
		if( i == writes.size() )
			i = -1;
		
		return i;
	}
	
	/** find any writes entries whose replies have probably been dropped
	 * 
	 * @param didx -- start index
	 * @param idx -- end index
	 * @param now -- current relative time
	 * @return index or -1 if none
	 */
	protected int findDropped( int didx, int idx, int now )
	{
		int i;
		
		// scrod the iterators, full indexing ahead
		for( i=didx; i<idx; ++i )
			if( ((wmd)writes.get(i)).oldTS(now) )
				break;
			
		if( i >= idx )
			i = -1;
		
		return i;
	}
	
	// write message data to store in Vector
	class wmd
	{
		int start;		// write start ticks (relative -- from start of run)
		int time;		// ticks until write complete
		boolean acked;	// true if we got an ACK for write
		int devid;		// device we sent to
		int seqnum;		// message sequence number
		
		// init a new one
		wmd( int s, int t, boolean a, int d, int n )
		{
			start = s;		// write start ticks (relative -- from start of run)
			time = t;		// ticks until complete
			acked = a;	// true if we got an ACK for write
			devid = d;		// device we sent to
			seqnum = n;		// message sequence number
		}
	
		/** return true if s matches our seqnum
		 * 
		 * @param s
		 */
		boolean compSN( int s )
		{
			return s == seqnum;
		}
	
		/** return true if s matches our seqnum
		 * 
		 * @param s
		 */
		boolean comp( int s, int i )
		{
			return (s == seqnum) && (i == devid);
		}
	
		/** return true if our start is way out of date vis t
		 *     (define way....)
		 * 
		 * @param t
		 */
		boolean oldTS( int t )
		{
			// for now if we started over a second ago....
			return t > (start+1000);
		}
	}
	
	// accumulated stats for messages
	class logaccum
	{
		int devid = -1;		// device we sent to (-1==not used)
		int starttime = Integer.MAX_VALUE; // first message time stamp
		int lasttime = 0;   // last logged message time stamp
		int messages = 0;	// total number of messages sent to dev
		int dropped = 0;		// number of msgs that were dropped (no reply)
		int notacked = 0;	// number of msgs that got no ACK to write
		int totWRtime = 0;	// total ticks for msg write
		int minWRtime = Integer.MAX_VALUE;	// minimum ticks for msg write
		int maxWRtime = 0;	// maximum ticks for msg write
		int totMStime = 0;	// total ticks for msg cycle (excluding dropped)
		int minMStime = Integer.MAX_VALUE;	// minimum ticks for msg cycle
		int maxMStime = 0;	// maximum ticks for msg cycle
		int totRSSI = 0;	// total received msg strength
		int noacks = 0;		// reMote missed ack counter
		int nomsgs = 0;		// reMote missed message counter
		
		int nomadj = 1;	// amount to adjust serialnum counts
		
		public logaccum() {}
		
		/** create an accumulator
		 * 
		 * @param bc -- doing a broadcast test if true
		 * @param nd -- number of devices we are testing
		 */
		public logaccum( boolean bc, int nd )
		{
			// Because the serial number of the Status Request 
			//  is incremented on each message we need to make some kind
			//  of adjustment to the returned nomsgs counts...
			// The nomsgs field in the Status Reply is just the
			//  difference between the last and current serial numbers.
			// So...if we are doing a broadcast send, the SN's are
			//  sequential for everyone and thus a diff of 1 means
			//  that nothing was missed.
			//  But, if we're doing a Round-Robin send/recv thing
			//  then the SN is incremented by the number of devices
			//  we are polling, so a diff of numdev means that nothing
			//  was missed.
			if( bc == true )
				nomadj = 1;
			else
				nomadj = nd;
		}
		
		// save common stuff
		void storeCommon( wmd d )
		{
			devid = d.devid;	// just in case we never set it
		
			// catch the first and last message times
			if( d.start < starttime )
				starttime = d.start;
			
			if( d.start > lasttime )
				lasttime = d.start;	// this'll be the last lasttime
			
			++messages;
			notacked += (d.acked ? 0 : 1);
			totWRtime += d.time;
			
			if( d.time < minWRtime )
				minWRtime = d.time;
			
			if( d.time > maxWRtime )
				maxWRtime = d.time;
		}
	
		// save stuff from failed message cycle
		public void storeFail( wmd d )
		{
			storeCommon( d );
			++dropped;
		}
	
		// save stuff from successful message cycle
		void storeSucc( wmd d, int mt, RoboStatMsg msg )
		{
			storeCommon( d );
			
			totMStime += mt;
			
			if( mt < minMStime )
				minMStime = mt;
			
			if( mt > maxMStime )
				maxMStime = mt;
			
			totRSSI += msg.rssi;
			
			noacks += msg.noacks;
			
			// kludge upon kludge
			// sometimes the first SN doesn't line up right so this
			//  tries to eliminate the outliers at the beginning.
			// if we missed more than 5 msgs we're probably hosed anyway...
			int nm = (msg.nomsgs / nomadj) - 1;
			if( (nm > 0) && (nm < 5) )
					nomsgs += nm;
		}
		
		public String makeString()
		{
			if( devid == -1 ) return "";	// nuthin' to see here...
			
			StringBuffer sb = new StringBuffer(100);
			
			if( !didhdr )
			{
				sb.append(
			 "\n\tdev msec msgs m/s !ack %!ack drop %drop " +
			 "minWR avgWR maxWR minMS avgMS maxMS avgRSSI " +
			 "noack % nomsg % devs bcast delay state txstr\n" );
				didhdr = true;
			}
			
			int time = lasttime - starttime;	// milli-sec duration
			int msgsucc = messages - dropped;
			
			// if we didn't get anything just forget it...
			if( msgsucc == 0 )
			{
				sb.append( "ACM\tTest Failed -- no successful messages!" );
			}
			else
			{
				sb.append( "ACM\t" );
				sb.append( devid + "\t" + time + "\t" + messages + "\t" );
				sb.append( dfmt(((double)(messages * 1000) / time))  + "\t" );
				sb.append( notacked + "\t" + dfmt((((double)notacked) / messages) * 100) + "\t" );
				sb.append( dropped + "\t" + dfmt((((double)dropped) / messages) * 100) + "\t" );
				sb.append( minWRtime + "\t" + (totWRtime / messages) + "\t" + maxWRtime + "\t" );
				sb.append( minMStime + "\t" + (totMStime / msgsucc) + "\t" + maxMStime + "\t" );
				sb.append( (totRSSI / msgsucc) + "\t" );
				sb.append( noacks + "\t" + dfmt((((double)noacks) / messages) * 100) + "\t" );
				sb.append( nomsgs + "\t" + dfmt((((double)nomsgs) / messages) * 100) + "\t" );
				// params from enclosing object
				sb.append( numdevs - (bcast ? 1 : 0) + "\t" );
				sb.append( bcast + "\t" );
				sb.append( autodelay + "\t" );
				sb.append( runstate + "\t" );
				sb.append( ((txstrength  == 0) ? 255 : txstrength) + "\t" );
				
				// create aggregate accumulation too
				add2Agg();
			}
			
			return sb.toString();
		}
		
		/** add our values in the 'global' aggAccum object
		 */
		protected void add2Agg()
		{
			aggAccum.devid = 0;
			aggAccum.lasttime = ( (aggAccum.lasttime > lasttime) ? aggAccum.lasttime : lasttime );
			aggAccum.starttime = ( (aggAccum.starttime < starttime) ? aggAccum.starttime : starttime );
			aggAccum.dropped += dropped;
			aggAccum.messages += messages;
			aggAccum.notacked +=  notacked;
			
			aggAccum.minWRtime = ( (aggAccum.minWRtime < minWRtime) ? aggAccum.minWRtime : minWRtime );
			aggAccum.totWRtime += totWRtime;
			aggAccum.maxWRtime = ( (aggAccum.maxWRtime > maxWRtime) ? aggAccum.maxWRtime : maxWRtime );
			
			aggAccum.minMStime = ( (aggAccum.minMStime < minMStime) ? aggAccum.minMStime : minMStime );
			aggAccum.totMStime += totMStime;
			aggAccum.maxMStime = ( (aggAccum.maxMStime > maxMStime) ? aggAccum.maxMStime : maxMStime );
			
			aggAccum.totRSSI += totRSSI;
			aggAccum.noacks += noacks;
			aggAccum.nomsgs += nomsgs;
			
			return;
		}
		
	} //end'o'logaccum
	

	/** convenient formatter for percents and stuff to keep them
	 *   from being obnoxious...
	 */
	static protected final int FMTNUMDIGITS = 2;
    static protected NumberFormat nf = NumberFormat.getInstance();
    {
    	// static initializer....
        nf.setMaximumFractionDigits( FMTNUMDIGITS );
        nf.setMinimumFractionDigits( FMTNUMDIGITS );
    }

	/** format doubles to fixed number of decimal places for easy reading
	 */
    public static String dfmt( double d )
    {
        return nf.format( d );
    } 
} // end'o'LogMessageTest
