/* HSMotorsM.nc --
 *  Code to use Timer1A and 1B to control Pulse Width Modulation 
 *   for modified Hobby Servo motors including added encoders.
 *   MotorA is the Left and MotorB is the Right.
 *   Assumes that they are mounted mirror image so have to turn
 *    in opposite directions to go fwd/bkd.
 *   Uses Frequency Generator outputs on PWM1a,b (ports PB5,6).
 *
 *  Note: encoder speed control should be enabled before a
 *          motion start command is issued, just in case.
 * Authors:		M.Schippling
 */


module HSMotorsM
{
  provides interface HSMotors;
  
  uses
  {
  	 interface HPLT1pwm as Pwm;
  }
}
implementation
{
	// current pulse-width settings for each motor
	uint8_t mApw = 0;
	uint8_t mBpw = 0;
	// number of encoder clicks for each motor
	//  note limited to ~127 cm travel at 2 click/cm
	uint8_t mAclicks = 0;
	uint8_t mBclicks = 0;
	enum { mAcw = 0x01, mBcw = 0x02, // motor direction, 0=CCW, 1==CW
		   mAlock = 0x04, mBlock= 0x08, // have speed lock, 1=yup
		   mtrun = 0x40,			 // motors are running, 1==yup
		   spdCntl = 0x80 };	     // enable speed control, 1==do it
	uint8_t mstate = 0;		// bits for direction, spdcntl, etc
	
	// Speed is specified as an 8-bit number from 4-255 which sets
	//   the relative On/Off pulse-width ratio for both motors.
	// The initial duty-cycle pw values are calculated from below.
	enum
	{
		MAXCCW		=	150,	// max CCW rotation count
		MINCCW		=	93,		// min CCW rotation count
		S0			=	90,		// ~1.5ms nominal servo off
		MAXCW		=	30,		// max CW rotation count
		MINCW		=	87,		// min CW rotation count
		PULSEROT	=	44,		// encoder pulses per wheel rotation
		WHEELCIRC	=	22,		// circumfrunce of wheel in CM
		// relative speed to duty cycle range divisor (...val=4...)
		SD	=	(512/(MAXCCW-MAXCW)),
		// encoder pulses per CM travel
		PC	=	(PULSEROT/WHEELCIRC),
		// milli-sec in 1 second...for convenience
		MS	=	1000,
		// number of clicks to wait for accelleration
		ACCELCLICK = 4,
		// increment value for speed control jogging 
		JOGINCR = 1
	};
	
	// map of relative speed to actual pulsewidth settings
	//  for both motors going both directions
	//  nominal forward is CCW motion for motorA (on left side)
	//                and   CW motion for motorB (on right side),
	//    because one is mirror mounted and needs to run "backward"...
	typedef struct
	{
		uint8_t spdset;		// requested speed setting, table index
		uint8_t count;		// desired encoder tick count for speed in ms
		uint8_t incr;		// count adjust threshold (% of count)
		uint8_t cwAcnt;		// CW MotorA pulse-width count setting
		uint8_t ccwAcnt;	// CCW MotorA pulse-width count setting
		uint8_t cwBcnt;		// CW MotorB pulse-width count setting
		uint8_t ccwBcnt;	// CCW MotorB pulse-width count setting
	} speed_table;
	
	// calculate based on:  pulses per wheel rotation
	//   count -- number of milli-sec between pulses @ each speed
	//				1000 / (pulses/cm * cm/sec)
	//   cnts  -- PW settings for each motor@direction
	//				seat'o'pants est based on SD range divisor and spdsetting
	//            nominal 90==0-mid-point >90==CCW, <90==CW rotation
	//    Note that RoboII mtrs/wheels/bats poop at about 25cm/sec
	speed_table spdtbl[] =
	{
		{   0, 0, 0, S0, S0, S0, S0 },						//  0cm/s -- calib 
		{  15, MS/(PC* 7), 10,								//  7cm/s
			 S0-(16/SD), S0+(16/SD), S0-(16/SD), S0+(16/SD) },
		{  20, MS/(PC*10),  5,								// 10cm/s
			 S0-(20/SD), S0+(20/SD), S0-(20/SD), S0+(20/SD) },
		{  25, MS/(PC*12),  5,								// 12cm/s
			 S0-(25/SD), S0+(25/SD), S0-(25/SD), S0+(25/SD) },
		{  50, MS/(PC*20),  1,								// 20cm/s
			 S0-(50/SD), S0+(50/SD), S0-(50/SD), S0+(50/SD) },
		{ 100, MS/(PC*25),  1,								// 25cm/s
			 S0-(100/SD), S0+(100/SD), S0-(100/SD), S0+(100/SD) }
	};
		/** nominal forward() settings, A-CCW, B-CW:
			0 = PWMsetABpw8( 90, 90 )
			50 = PWMsetABpw8( 102, 78 )
			100 = PWMsetABpw8( 115, 65 )
			150 = PWMsetABpw8( 127, 53 )
			200 = PWMsetABpw8( 140, 40 )
		**/
	
	speed_table *curtbl;		// current speed_table entry
	
	  
	/** init()
	*  setup the Timer1A,B for Phase and Frequency correct PWM
	**/
	command result_t HSMotors.init()
	{
		call Pwm.init();
		//call Pwm.setABpw8( 102, 78 );		// force motor test
		return SUCCESS;
	}
	
	/** stopMotors()
	*    Stop both the PWM outputs and set outputs to 0.
	*  @return nada.
	*/
	command void HSMotors.stopMotors()
	{
		mstate &= ~mtrun;
		call Pwm.stopAll();
		
		// save new motor PWM values if enabled
		if( (mstate & mAlock) )
			if( (mstate & mAcw) )
				curtbl->cwAcnt = mApw;
			else
				curtbl->ccwAcnt = mApw;
		
		if( (mstate & mBlock) )
			if( (mstate & mBcw) )
				curtbl->cwBcnt = mBpw;
			else
				curtbl->ccwBcnt = mBpw;
	}
	
	/** Startup the motors at specified speed seeting.
	 **  find speed_table entry,
	 **  set curtbl based on given speed setting
	 **  reset mA,Bclicks = 0
	 **  set mA,Bpw globals to values from curtbl entry
	 **      using mstate bits to select directions
	 ** set PWM values to mA,Bpw values
	**/
	void setPW( uint8_t s )
	{
		int i;
		mAclicks = 0;
		mBclicks = 0;
		
		// set curtbl to point to the speed setting entry in spdtbl
		// note drops out on last spdtbl entry if not found
		//  so last entry will be the default
		for( i = 0;
			 i < (sizeof(spdtbl) / sizeof(speed_table)) - 1 ;
			 ++i )
		{
			if( s <= spdtbl[i].spdset )
				break;
		}
		curtbl = &spdtbl[i];
	
		// get pulse-width settings based on pre-set direction bits
		mApw = ((mstate & mAcw)) ? curtbl->cwAcnt : curtbl->ccwAcnt;
		mBpw = ((mstate & mBcw)) ? curtbl->cwBcnt : curtbl->ccwBcnt;
	
		// enable speed control
//		mstate |= (spdCntl + mtrun);
		mstate |= mtrun;

		// clear lock indicators
		mstate &= ~(mAlock + mBlock);
		
		// start 'em up
		call Pwm.setABpw8( mApw, mBpw );
		//call Pwm.setABpw8( 90, 90 );
	}
	
	/** forward()
	*    Start robot moving forward at the given relative speed.
	*  @return nada.
	*/
	command void HSMotors.forward( uint8_t s )
	{
		// set A=CCW, B=CW
		mstate &= ~mAcw;
		mstate |= mBcw;
		setPW( s );
	}
	
	/** backward()
	*    Start robot moving backward at the given relative speed.
	*  @return nada
	*/
	command void HSMotors.backward( uint8_t s )
	{
		// set A=CW, B=CCW
		mstate |= mAcw;
		mstate &= ~mBcw;
		setPW( s );
	}
	
	/**
	* rotateCW
	*    Start robot rotating to the right (clock-wise) at the given speed.
	*  @return nada
	*/
	command void HSMotors.rotateCW( uint8_t s )
	{
		// set A=CCW, B=CCW
		mstate &= ~mAcw;
		mstate &= ~mBcw;
		setPW( s );
	}
	
	/**
	* rotateCCW
	*    Start robot rotating to the left (ccw) at the given speed.
	*  @return nada
	*/
	command void HSMotors.rotateCCW( uint8_t s )
	{
		// set A=CW, B=CW
		mstate |= mAcw;
		mstate |= mBcw;
		setPW( s );
	}
	
	/** jogASpeed()
	*    Jog Left -- adjust MotorA speed (pulsewidth setting) by given amount.
	*    silently limits at MIN/MAX values
	*  @return nada.
	*/
	command void HSMotors.jogASpeed( int8_t d )
	{
		mApw += d;
		// enforce limits
		if( (mstate & mAcw) )	// CW
		{
			if( mApw < MAXCW )
				mApw = MAXCW;
			else if( mApw > MINCW )
				mApw = MINCW;
		}
		else					// CCW
		{
			if( mApw > MAXCCW )
				mApw = MAXCCW;
			else if( mApw < MINCCW )
				mApw = MINCCW;
		}
		
		call Pwm.setApw8( mApw );
	}
	
	/** jogBSpeed()
	*    Jog Right-- adjust MotorB speed (pulsewidth setting) by given amount.
	*    silently limits at MIN/MAX values
	*  @return nada.
	*/
	command void HSMotors.jogBSpeed( int8_t d )
	{
		mBpw += d;
		// enforce limits
		if( (mstate & mBcw) )	// CW
		{
			if( mBpw < MAXCW )
				mBpw = MAXCW;
			else if( mBpw > MINCW )
				mBpw = MINCW;
		}
		else					// CCW
		{
			if( mBpw > MAXCCW )
				mBpw = MAXCCW;
			else if( mBpw < MINCCW )
				mBpw = MINCCW;
		}
		
		call Pwm.setBpw8( mBpw );
	}
	
	/** enableSpdCntl()
	*    En/Dis-able encoder driven speed control in check*Spd() methods.
	*	TRUE enables, FALSE disables.
	*	Used to take motor control during Target Tracking.
	*  @return nada.
	*/
	command void HSMotors.enableSpdCntl( bool en )
	{
		if( en )
			mstate |= spdCntl;
		else
			mstate &= ~spdCntl;
	}
		
	/** checkASpeed()
	*    Check the Left--MotorA speed against given count value.
	*	Adjust MotorA pulse width if necessary
	*	 and save new pw setting for use later in speed table.
	*	Also increments distance count so we can check how far we went.
	*  @return nada
	*/
	command void HSMotors.checkASpeed( uint8_t count )
	{
		int8_t jog;
		++mAclicks;
		
		// adjust motor if enabled and we are over the accel phase
		if( ((mstate & (spdCntl | mtrun)) == (spdCntl | mtrun)) &&
			(mAclicks > ACCELCLICK) )
		{
			mstate &= ~mAlock;	// clear lock indicator
			// sign of change depends on direction of rotation
			//  CW gets smaller to speed up
			jog = ((mstate & mAcw)) ? -JOGINCR : JOGINCR;
			if( count < (curtbl->count - curtbl->incr) )	// running fast
				call HSMotors.jogASpeed( -jog );	// slow down
			else if( count > (curtbl->count + curtbl->incr) )
				call HSMotors.jogASpeed( jog );
			else
				mstate |= mAlock;		// signal that we have a good setting
		}
	}
	
	/** checkBSpeed()
	*    Check the Right--MotorB speed against given count value.
	*	Adjust MotorB pulse width if necessary
	*	 and save new pw setting for use later in speed table.
	*	Also increments distance count so we can check how far we went.
	*  @return nada
	*/
	command void HSMotors.checkBSpeed( uint8_t count )
	{
		int8_t jog;
		++mBclicks;
		
		// adjust motor if enabled
		if( ((mstate & (spdCntl | mtrun)) == (spdCntl | mtrun)) &&
			(mBclicks > ACCELCLICK) )
		{
			mstate &= ~mBlock;	// clear lock indicator
			// sign of change depends on direction of rotation
			//  CW gets smaller to speed up
			jog = ((mstate & mBcw)) ? -JOGINCR : JOGINCR;
			if( count < (curtbl->count - curtbl->incr) )	// running fast
				call HSMotors.jogBSpeed( -jog );	// slow down
			else if( count > (curtbl->count + curtbl->incr) )
				call HSMotors.jogBSpeed( jog );
			else
				mstate |= mBlock;
		}
	}
	
	/** getDistance()
	*  @return the distance traveled in clicks during this motion.
	*			Note: this will be calculated from the sum of the
	*			 click counts from both wheels. It will always be
	*			 positive, even during rotations.
	*			Note: value is ~4*CM traveled because it's really just the
	*			 number of encoder clicks that are reported when using
	*			 check*Spd() and clicks happen about 2/CM.
	*/
	command uint16_t HSMotors.getDistance()
	{
		return ((uint16_t) mAclicks + mBclicks);
	}
	
} // end'o'impl
