/* HPLT3captureM.nc -- Code to use Timer3 to time pulses on port PE7.
 *	Defaults to values for 16 bit resolution of 1.088us per bit.
 *
 * Authors:		M.Schippling
 */

module HPLT3captureM
{
  provides interface HPLT3capture;
}
implementation
{
	uint8_t t3bmask = 0;	// added bits to mask with TCCR3B prescale set

// assign input pin name
TOSH_ASSIGN_PIN(TRIGGER, E, 7);	// port,pin E7
 
  /** init()
   *  Setup the Timer3 for normal up count, mode 0.
   *  Initialize interrupt time capture on PE7/INT3 (or in real AVR INT7).
   *  Use capture edge specified by edge argument, where true=rising.
   *   default settings: /8 internal clock pre-scale
   *					 ICR3 as capture count trigger
   *					 OCR3 not used
   *					 max count is approx 71ms
  **/
  command result_t HPLT3capture.init( bool edge )
  {
    // Set input pin direction and turn on internal pullup
    TOSH_MAKE_TRIGGER_INPUT();
    TOSH_SET_TRIGGER_PIN();

	// configure count interrupt edge
	if( edge == TRUE )
		t3bmask = (1 << ICES3);

	// turn off OCR3 output, set WGM3 to mode 0,  and noise canceler off
    outp( 0, TCCR3A );	// clear OCR3 outputs and lowWGM bits
    outp( t3bmask, TCCR3B );	// set edge, clock-select, and highWGM bits

	// Set prescaler to CK/8
    call HPLT3capture.setPreScale( 0x02 );

    // setup interrupt
    call HPLT3capture.intEnable();

    return SUCCESS;
  }

  /** setPreScale()
   *  Set the clock pre-scale and clock source value
   * @param ps:	TCCR3B register set value.
  *		0 -- 000 No clock source. (Timer/Counter stopped)
   *		1 -- 001 clkI/O/1    136ns clock (clock no-prescaling)
   *		2 -- 010 clkI/O/8 	   1.088us
   *		3 -- 011 clkI/O/64     8.704us
   *		4 -- 100 clkI/O/256   34.816ms
   *		5 -- 101 clkI/O/1024 139.264ms
   *		6 -- 110 External clock source on Tn pin. Clock on falling edge
   *		7 -- 111 External clock source on Tn pin. Clock on rising edge
   * @return nada.
   **/
  command void HPLT3capture.setPreScale( uint8_t ps )
  {
  	// OR with edge select bit, as per init(), and save for later
	t3bmask &= ~0x07;
	t3bmask |= (ps & 0x07);
	outp( t3bmask, TCCR3B );

	return;
  }

  /** invertEdge()
   *   Invert the edge detect sense, if it was 1 it's 0 after, USW
   **/
  command void HPLT3capture.invertEdge()
  {
  	// diddle edge select bit and write reg bit
	t3bmask ^= (1 << ICES3);
	if( t3bmask & (1 << ICES3) )
		sbi( TCCR3B, ICES3 );
	else
		cbi( TCCR3B, ICES3 );

	return;
  }

  /** enable and disable interrupts **/
  async command void HPLT3capture.intDisable()
  { cbi(ETIMSK, TICIE3); }
  async command void HPLT3capture.intEnable()
  { sbi(ETIMSK, TICIE3); }

  /** fire()
   *  Event sent when the input interrupt goes off.
   *  Gets the count value and a counter overflow indicator.
   *  If over==true counter wrapped between last call and this.
   **/
  default async event result_t HPLT3capture.fire( uint16_t count, bool over )
  { return SUCCESS; }
  
  TOSH_INTERRUPT(SIG_INPUT_CAPTURE3)
  {
	bool over;
  	// read count registers
    uint16_t count = inp( ICR3L );
	count |= (inp( ICR3H ) << 8);

	// clear int
	sbi( ETIFR, ICF3 );

	// get overflow situation on TOV3
	over = (inp( ETIFR ) & (1 << TOV3) ) ? TRUE : FALSE ;

	// clear overflow and counters
	sbi( ETIFR, TOV3 );
	outp( 0, TCNT3H );
	outp( 0, TCNT3L );
	
	// call user
    signal HPLT3capture.fire( count, over );
  }

} // end'o'impl
